/* ------------------------------------------------------------------------------
  File: chr6d_DCM.h
  Author: Vassilis Varveropoulos
  Version: 1.0

  Description: Functions for calculating Direction Cosine Matrix (DCM)
------------------------------------------------------------------------------ */
#include "stdio.h"
#include "stm32f10x.h"
#include "DCM_test.h"

#define KpXY 0.0001
#define KiXY 0.0000005
#define KpZ 0.5
#define KiZ 0.0005
#define PI 3.14159265359

void DCMNormalize(void);
void DCMDriftCorrect(void);
float VectorDotProduct(float* v1, float* v2);
void VectorCrossProduct(float* vOut, float* v1,float* v2);
void VectorScale(float* vOut, float* v, float scale);
void VectorAdd(float* vOut, float* v1, float* v2);
void MatrixMultiply(float mOut[3][3], float m1[3][3], float m2[3][3]);
void MatrixAdd(float mOut[3][3], float m1[3][3], float m2[3][3]);

float gDCMMatrix[3][3] = {
    { 1,0,0 },
    { 0,1,0 },
    { 0,0,1 }
};
float DCM1;



float Gyros[3] = {0, 0, 0};
float Accels[3] = {0, 0, 0};
float Omega[3] = {0, 0, 0};
float OmegaI[3] = {0, 0, 0};
float OmegaP[3] = {0, 0, 0};
float ErrorXY[3] = {0, 0, 0};
float ErrorZ[3] = {0, 0, 0};


void DCMUpdate(void)
{
    float DCMMatrixTmp[3][3];
    float DCMMatrixUpdate[3][3];
    float Ts;

    // Calculate period
    Ts = 0.02;

    // Update gyros vector and apply signs
    Gyros[0] = -GyroCon[0]; // x gyro gConfig.x_gyro_bias
    Gyros[1] = -GyroCon[1]; // y gyro
    Gyros[2] = -GyroCon[2]; // z gyro
    Accels[0] = AccCon[0]; // x accel
    Accels[1] = -AccCon[1]; // y accel
    Accels[2] = -AccCon[2]; // z accel

    // Add proportional and integral terms
    VectorAdd(&Omega[0], &Gyros[0], &OmegaI[0]);
    VectorAdd(&Omega[0], &Omega[0], &OmegaP[0]);

    // Adjust for centrifugal acceleration
    // TBD - Speed required

    // Create update DCM matrix
    DCMMatrixUpdate[0][0] = 0;
    DCMMatrixUpdate[0][1] = -Omega[2]*Ts; //-z
    DCMMatrixUpdate[0][2] = Omega[1]*Ts; //y
    DCMMatrixUpdate[1][0] = Omega[2]*Ts; //z
    DCMMatrixUpdate[1][1] = 0;
    DCMMatrixUpdate[1][2] = -Omega[0]*Ts; //-x
    DCMMatrixUpdate[2][0] = -Omega[1]*Ts; //-y
    DCMMatrixUpdate[2][1] = Omega[0]*Ts; //x
    DCMMatrixUpdate[2][2] = 0;

    // Update main DCM matrix
    MatrixMultiply(&DCMMatrixTmp[0], &gDCMMatrix[0], &DCMMatrixUpdate[0]);
    MatrixAdd(&gDCMMatrix[0], &gDCMMatrix[0], &DCMMatrixTmp[0]);

    // Normalize matrix
    DCMNormalize();

    // Correct for drift
    DCMDriftCorrect();
    DCM1 = gDCMMatrix[2][2];

}

void DCMNormalize(void)
{
    float Xor[3] = {0, 0, 0};
    float Yor[3] = {0, 0, 0};
    float Zor[3] = {0, 0, 0};
    float error, renorm;

    error = -VectorDotProduct(&gDCMMatrix[0][0],&gDCMMatrix[1][0]) * 0.5; // eq.18

    VectorScale(&Xor[0], &gDCMMatrix[1][0], error); //eq.19
    VectorAdd(&Xor[0], &Xor[0], &gDCMMatrix[0][0]); //eq.19

    VectorScale(&Yor[0], &gDCMMatrix[0][0], error); //eq.19
    VectorAdd(&Yor[0], &Yor[0], &gDCMMatrix[1][0]); //eq.19

    VectorCrossProduct(&Zor[0],&Xor[0],&Yor[0]); //eq.20

    renorm = .5 *(3 - VectorDotProduct(&Xor[0],&Xor[0])); //eq.21
    VectorScale(&gDCMMatrix[0][0], &Xor[0], renorm);

    renorm = .5 *(3 - VectorDotProduct(&Yor[0],&Yor[0])); //eq.21
    VectorScale(&gDCMMatrix[1][0], &Yor[0], renorm);

    renorm = .5 *(3 - VectorDotProduct(&Zor[0],&Zor[0])); //eq.21
    VectorScale(&gDCMMatrix[2][0], &Zor[0], renorm);

}

void DCMDriftCorrect(void)
{
    float OmegaITmp[3];
    // Roll and Pitch drift correction
    VectorCrossProduct(&ErrorXY[0], &gDCMMatrix[2][0], &Accels[0]);
    VectorScale(&OmegaP[0], &ErrorXY[0], KpXY);
    VectorScale(&OmegaITmp[0], &ErrorXY[0], KiXY);
    VectorAdd(&OmegaI[0], &OmegaI[0], &OmegaITmp[0]);

    // Yaw drift correction
    // TBD - Requires mag
}



float VectorDotProduct(float* v1, float* v2)
{
    float dp = 0;
    int n;
    for(n = 0; n < 3; n++)
    {
        dp += v1[n]*v2[n];
    }
    return dp;
}

void VectorCrossProduct(float* vOut, float* v1,float* v2)
{
    vOut[0] = (v1[1]*v2[2]) - (v1[2]*v2[1]);
    vOut[1] = (v1[2]*v2[0]) - (v1[0]*v2[2]);
    vOut[2] = (v1[0]*v2[1]) - (v1[1]*v2[0]);
}

//Multiply the vector by a scalar.
void VectorScale(float* vOut, float* v, float scale)
{
    int n;
    for(n = 0; n < 3; n++)
    {
        vOut[n] = v[n]*scale;
    }
}

void VectorAdd(float* vOut, float* v1, float* v2)
{
    int n;
    for(n = 0; n < 3; n++)
    {
        vOut[n] = v1[n] + v2[n];
    }
}

void MatrixMultiply(float mOut[3][3], float m1[3][3], float m2[3][3])
{
    float tmp[3];
    int x, y, w;

    for(x = 0; x < 3; x++)
    {
        for (y = 0; y < 3; y++)
        {
            for(w = 0; w < 3; w++)
            {
                tmp[w] = m1[x][w] * m2[w][y];
            }
            mOut[x][y] = tmp[0] + tmp[1] + tmp[2];
        }
    }
}

void MatrixAdd(float mOut[3][3], float m1[3][3], float m2[3][3])
{
    int x, y;
    for (x = 0; x < 3; x++)
    {
        for (y = 0; y < 3; y++)
        {
            mOut[x][y] = m1[x][y] + m2[x][y];
        }
    }
}
